import React, { createContext, useEffect, useState } from "react";
import { createRoot } from "react-dom/client";
import { BrowserRouter as Router, Routes, Route } from "react-router-dom";
import useWebSocket from "react-use-websocket";
import axios from "axios";

import { Footer, Header } from "./components";
import { HomePage, RestaurantPage, NotFoundPage, AdminPage } from "./views";

import "bootstrap/dist/css/bootstrap.min.css";
import "./styles/index.css";

const socketUrl = `${process.env.REACT_APP_SOCKET_URL}/ws`;
export const User = createContext(null);

export default function App() {
  const storedSession = localStorage.getItem("user");
  const [restaurantsList, setRestaurantsList] = useState([]);
  const [selection, setSelection] = useState(null);
  const [loading, setLoading] = useState(true);
  const [reload, setReload] = useState(true);
  const [user, setUser] = useState(storedSession ? JSON.parse(storedSession) : null);
  const { lastMessage } = useWebSocket(socketUrl, {
    shouldReconnect: () => true,
  });

  useEffect(() => {
    if (reload) {
      axios
        .get(`${process.env.REACT_APP_BASE_URL_BACK}/restaurants`)
        .then((res) => {
          setRestaurantsList(res.data);
          setLoading(false);
          setReload(false);
        })
        .catch((e) => {
          console.log(e);
          setLoading(false);
          setReload(false);
        });
    }
  }, [reload]);

  useEffect(() => {
    window.addEventListener("popstate", () => setReload(true));

    let path = window.location.pathname.split("/");
    if (restaurantsList && path.length >= 2 && path[1]) {
      let name = decodeURIComponent(path[1]);
      let filter = restaurantsList.filter((restaurant) => restaurant.name == name);
      if (filter) {
        setSelection(filter[0]);
      }
    } else {
      setSelection(null);
    }

    return () => window.removeEventListener("popstate", () => setReload(true));
  }, [restaurantsList]);

  useEffect(() => {
    if (lastMessage?.data) {
      let new_message = JSON.parse(lastMessage.data);
      if (new_message.type == "data") {
        setReload(true);
      }
    }
  }, [lastMessage]);

  return (
    <div className="app">
      <User.Provider value={[user, setUser]}>
        <Router>
          <Header {...{ selection, setSelection }} />
          <div className="page">
            <Routes>
              <Route
                exact
                path="/"
                element={<HomePage {...{ restaurantsList, setSelection, loading }} />}
              />
              {user?.admin && (
                <Route path="/admin" element={<AdminPage lastmessage={lastMessage} />} />
              )}
              <Route
                path="/:restaurant"
                element={<RestaurantPage {...{ selection, setSelection, lastMessage }} />}
              />
              <Route path="*" element={<NotFoundPage />} />
            </Routes>
          </div>
        </Router>
        <Footer />
      </User.Provider>
    </div>
  );
}

const container = document.getElementById("root");
const root = createRoot(container);
root.render(<App />);
