from fastapi import APIRouter, Cookie, Depends, HTTPException
from sqlalchemy.orm import Session
from typing import List

from db import schemas, crud
from db.database import get_db


router = APIRouter(prefix="/api", tags=["timetable"])


# Manage opening hours

@router.get('/{place}/opening_hours', response_model=List[schemas.OpeningHours])
async def get_opening_hours(place: str, db: Session = Depends(get_db)):
    return crud.get_opening_hours(place, db)


@router.post('/opening_hours', response_model=schemas.OpeningHours)
async def create_opening_hours(opening_hours: schemas.OpeningHoursBase, db: Session = Depends(get_db)):
    return crud.create_opening_hours(opening_hours, db)


@router.delete('/opening_hours/{id}', response_model=None)
async def delete_opening_hours(id: int, db: Session = Depends(get_db)):
    return crud.delete_opening_hours(id, db)


# Manage exceptional closure

@router.get('/{place}/closure', response_model=List[schemas.Closure])
async def get_closure(place: str, db: Session = Depends(get_db)):
    return crud.get_closure(place, db)


@router.post('/closure', response_model=schemas.Closure)
async def create_closure(closure: schemas.ClosureBase, connect_id: str = Cookie(...), db: Session = Depends(get_db)):
    user = crud.get_user(connect_id, db)
    if user.admin:
        return crud.create_closure(closure, db)
    else:
        raise HTTPException(status_code=403, detail="Administrator privilege required")


@router.delete('/closure/{id}', response_model=None)
async def delete_closure(id: int, db: Session = Depends(get_db)):
    return crud.delete_closure(id, db)


# Render restaurants infos

@router.get('/restaurants', response_model=List[schemas.Restaurant])
async def get_restaurants(db: Session = Depends(get_db)):
    return crud.get_restaurants(db)
