import React, { useEffect, useState } from "react";
import axios from "axios";
import {
  Line,
  Area,
  XAxis,
  YAxis,
  CartesianGrid,
  Tooltip,
  ResponsiveContainer,
  ComposedChart,
} from "recharts";

import "../styles/Graph.css";

const CustomTooltip = ({ active, payload, label }) => {
  return (
    <>
      {active && payload && payload.length && (
        <div className="custom-tooltip">
          <p className="label">{`${formatXAxis(label)}`}</p>
          <p className="label">{`Temps d'attente : ${payload[0].value} minutes`}</p>
        </div>
      )}
    </>
  );
};

function formatXAxis(value) {
  if (value == 0) return "";
  value = Math.round(value);
  return Math.floor(value / 60).toString() + "h" + (value % 60).toString().padStart(2, "0");
}

export default function Graph({ place, lastMessage }) {
  const [checked, setChecked] = useState(false);
  const [currentData, setCurrentData] = useState({ data: [], start: 0, end: 0 });
  const [avgData, setAvgData] = useState([]);
  const [reload, setReload] = useState(true);

  useEffect(() => {
    if (reload) {
      axios
        .get(
          `${process.env.REACT_APP_BASE_URL_BACK}/${encodeURIComponent(place)}/stats/current_graph`,
        )
        .then((response) => {
          setCurrentData(response.data);
        });

      axios
        .get(`${process.env.REACT_APP_BASE_URL_BACK}/${encodeURIComponent(place)}/stats/avg_graph`)
        .then((response) => {
          setAvgData(response.data);
        });

      setReload(false);
    }
  }, [reload]);

  useEffect(() => {
    if (lastMessage?.data) {
      let new_message = JSON.parse(lastMessage.data);
      if (new_message.type == "data") {
        setReload(true);
      }
    }
  }, [lastMessage]);

  return (
    <>
      {!!currentData?.data?.length && (
        <div style={{ height: "60%", padding: "3rem" }}>
          <div className="graph">
            <ResponsiveContainer width="100%" height="100%">
              <ComposedChart
                margin={{
                  top: 5,
                  right: 30,
                  left: -10,
                  bottom: 5,
                }}
              >
                <defs>
                  <linearGradient id="colorGradient" x1="0" y1="0" x2="0" y2="1">
                    <stop offset="10%" stopColor="#ff0000" stopOpacity={0.55} />
                    <stop offset="50%" stopColor="#fff200" stopOpacity={0.5} />
                    <stop offset="90%" stopColor="#1e9600" stopOpacity={0.35} />
                  </linearGradient>
                </defs>
                <CartesianGrid stroke="#FFFFFF" strokeDasharray="1 3" />
                <XAxis
                  axisLine={false}
                  tickLine={false}
                  tick={{ fill: "#FFFFFF", fontSize: "18" }}
                  ticks={[...Array(4).keys()].map(
                    (i) => currentData.start + (i * (currentData.end - currentData.start)) / 3,
                  )}
                  dataKey="name"
                  type="number"
                  interval="preserveStartEnd"
                  domain={[currentData.start, currentData.end]}
                  tickFormatter={formatXAxis}
                />
                <YAxis
                  axisLine={false}
                  tickLine={false}
                  tick={{ fill: "#FFFFFF", fontSize: "18" }}
                  tickInt
                  tickCount={10}
                  dataKey="time"
                  type="number"
                  domain={[0, (dataMax) => 10 * Math.floor((dataMax + 10) / 10)]}
                  allowDecimals={false}
                  name="Temps d'attente"
                />
                <Tooltip content={<CustomTooltip />} />
                <Area
                  data={currentData.data}
                  type="monotone"
                  dataKey="time"
                  stroke="#FFFFFF"
                  strokeWidth={2}
                  fillOpacity={1}
                  fill="url(#colorGradient)"
                />
                {checked ? (
                  <Line
                    data={avgData}
                    type="monotone"
                    dataKey="time"
                    stroke="#FF0000"
                    strokeWidth={2}
                    dot={false}
                  />
                ) : (
                  <div />
                )}
              </ComposedChart>
            </ResponsiveContainer>
          </div>
          <div className="graph-title">
            Temps d&apos;attente estimé depuis l&apos;ouverture (en minutes)
          </div>
          <button id="graph-avg-graph" onClick={() => setChecked(!checked)}>
            {checked ? "Retirer le temps d'attente moyen" : "Afficher le temps d'attente moyen"}
          </button>
        </div>
      )}
    </>
  );
}
