<?php

namespace BackendBundle\Controller;

use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Request;

class EntityBrowserController extends Controller
{
    /**
     * @param string $name
     * @return Response
     *
     * @Route("/admin/entity/{name}/list", requirements={"name" = "league|prediction"})
     */
    public function listAction(string $name)
    {
        $em = $this->getDoctrine()->getManager();
        $columns = [];

        $entityInfo = $em->getClassMetadata('BackendBundle:'.ucfirst($name));

        foreach (array_merge($entityInfo->getFieldNames(), $entityInfo->getAssociationNames()) as $field) {
            $columns[] = [
                'field' => $this->fromCamelCase($field),
                'type' => $entityInfo->getTypeOfField($field)
            ];
        }

        return $this->render(
            '@Backend/EntityBrowser/list.html.twig',
            [
                'columns' => $columns,
                'name' => $name,
            ]
        );
    }

    /**
     * @param string $name
     * @param Request $request
     * @return Response
     *
     * @Route("/admin/entity/{name}/new", requirements={"name" = "league|prediction"})
     */
    public function newAction(string $name, Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $entityInfo = $em->getClassMetadata('BackendBundle:'.ucfirst($name));
        $newElement =  $entityInfo->newInstance();
        $addForm = $this->createForm("BackendBundle\\Form\\add".ucfirst($name)."Type", $newElement);

        if ($request->getMethod() == 'POST') {
            $addForm->handleRequest($request);
            $repository = $em->getRepository('BackendBundle:'.ucfirst($name));
            if ($addForm->isValid()) {
                $alreadyExists = $repository->alreadyExists($newElement);
                if (!$alreadyExists) {
                    $em->persist($newElement);
                    $em->flush();
                    $this->get('session')->getFlashBag()->add('success', "L'élément a bien été ajouté");
                } else {
                    $this->get('session')->getFlashBag()->add('error', "Cet élément existe déjà");
                }
            }
        }

        return $this->render(
            '@Backend/EntityBrowser/form.html.twig',
            [
                'name' => $name,
                'form' => $addForm
            ]
        );
    }

    /**
     * @param string $name
     * @param int $id
     * @param Request $request
     * @return Response
     *
     * @Route("/admin/entity/{name}/edit/{id}", requirements={"name" = "league|prediction", "id"})
     */
    public function editAction(string $name, int $id, Request $request)
    {
        $em = $this->getDoctrine()->getManager();
        $repository = $em->getRepository('BackendBundle:'.ucfirst($name));
        $element = $repository->find($id);

        if ($element) {
            $editForm = $this->createForm("BackendBundle\\Form\\add".ucfirst($name)."Type", $newElement);
            if ($request->getMethod() == 'POST') {
                $editForm->handleRequest($request);
                if ($editForm->isValid()) {
                    $em->flush();
                    $this->get('session')->getFlashBag()->add('success', "L'élément a bien été ajouté");
                }
            }
        } else {
            $this->get('session')->getFlashBag()->add('error', "Vous tentez d'accéder à un élément qui n'existe pas");
            $this->redirectToRoute('backend_entitybrowser_list');
        }

        return $this->render(
            '@Backend/EntityBrowser/form.html.twig',
            [
                'name' => $name,
                'form' => $editForm
            ]
        );
    }

    /**
     * @param string $name
     * @param int $id
     * @return Response
     *
     * @Route("/admin/entity/{name}/delete/{id}", requirements={"name" = "league|prediction", "id"})
     */
    public function deleteAction(string $name, int $id)
    {
        $em = $this->getDoctrine()->getManager();
        $repository = $em->getRepository('BackendBundle:'.ucfirst($name));
        $element = $repository->find($id);

        $response = new Response();

        if ($element) {
            $em->remove($entity);
            $em->flush();
            $response->setStatusCode(200);
            $response->setContent([
                'success' => true,
                'message' => 'L\'élément a bien été supprimé'
            ]);
        } else {
            $response->setStatusCode(400);
            $response->setContent([
                'success' => false,
                'message' => 'Vous tentez d\'accéder à un élément qui n\'éxiste pas ou qui a déjà été supprimé'
            ]);
        }

        return $response;
    }

    /**
     * @param string $name
     * @return Response
     *
     * @Route("/admin/entity/list/{name}", requirements={"name" = "league|prediction"})
     */
    public function ajaxListAction(string $name)
    {
        $em = $this->getDoctrine()->getManager();
        $repository = $em->getRepository('BackendBundle:'.ucfirst($name));

        $elements = $repository->findAll();

        $serializer = $this->get('jms_serializer');

        $response = $serializer->serialize($elements,'json');

        if ($name == 'prediction') {
            $array = json_decode($response);
            foreach ($array as $prediction) {
                $prediction->league = $prediction->league->name;
            }
            $response = json_encode($array);
        }

        return new Response($response);

    }

    private function fromCamelCase($input) {
        preg_match_all('!([A-Z][A-Z0-9]*(?=$|[A-Z][a-z0-9])|[A-Za-z][a-z0-9]+)!', $input, $matches);
        $ret = $matches[0];
        foreach ($ret as &$match) {
            $match = $match == strtoupper($match) ? strtolower($match) : lcfirst($match);
        }
        return implode('_', $ret);
    }
}
